<?php
// ver_ventas.php
date_default_timezone_set('America/Guatemala'); // Ajusta la zona horaria según tu ubicación
include 'db_connection.php';
include 'sucursal_config.php';

// Obtenemos la fecha de hoy
$hoy = date('Y-m-d');

// 1. OBTENER NOMBRE DE SUCURSAL
$sucursalName = '';
$sqlSuc = "SELECT nombre FROM sucursales WHERE id = ? LIMIT 1";
$stmtSuc = $conn->prepare($sqlSuc);
$stmtSuc->bind_param("i", $sucursal_id);
$stmtSuc->execute();
$resSuc = $stmtSuc->get_result();
if ($rowS = $resSuc->fetch_assoc()) {
    $sucursalName = $rowS['nombre'];
}

// 2. Parámetros de paginación para Ventas Normales
$pageN = isset($_GET['pageN']) ? max(1, intval($_GET['pageN'])) : 1;
$limitN = 5;
$offsetN = ($pageN - 1) * $limitN;

// Parámetros de paginación para Ventas Especiales
$pageE = isset($_GET['pageE']) ? max(1, intval($_GET['pageE'])) : 1;
$limitE = 5;
$offsetE = ($pageE - 1) * $limitE;

// 3. Contar total de Ventas Normales del día
$sqlCountN = "SELECT COUNT(*) AS totalN
              FROM ventas
              WHERE DATE(fecha_y_hora) = ?
                AND sucursal_id = ?";
$stmtCN = $conn->prepare($sqlCountN);
$stmtCN->bind_param("si", $hoy, $sucursal_id);
$stmtCN->execute();
$resCN = $stmtCN->get_result();
$rowCN = $resCN->fetch_assoc();
$totalN = (int)$rowCN['totalN'];
$totalPagesN = ceil($totalN / $limitN);

// 4. Obtener Ventas Normales con paginación
$sqlN = "SELECT id, fecha_y_hora, total
         FROM ventas
         WHERE DATE(fecha_y_hora) = ?
           AND sucursal_id = ?
         ORDER BY fecha_y_hora DESC
         LIMIT ? OFFSET ?";
$stmtN = $conn->prepare($sqlN);
$stmtN->bind_param("siii", $hoy, $sucursal_id, $limitN, $offsetN);
$stmtN->execute();
$resN = $stmtN->get_result();
$ventas_normales = [];
while ($row = $resN->fetch_assoc()) {
    $ventas_normales[] = $row;
}

// 5. Contar total de Ventas Especiales del día
$sqlCountE = "SELECT COUNT(*) AS totalE
              FROM ventas_especiales
              WHERE DATE(fecha_y_hora) = ?
                AND sucursal_id = ?";
$stmtCE = $conn->prepare($sqlCountE);
$stmtCE->bind_param("si", $hoy, $sucursal_id);
$stmtCE->execute();
$resCE = $stmtCE->get_result();
$rowCE = $resCE->fetch_assoc();
$totalE = (int)$rowCE['totalE'];
$totalPagesE = ceil($totalE / $limitE);

// 6. Obtener Ventas Especiales con paginación
$sqlE = "SELECT id, fecha_y_hora, descripcion, precio
         FROM ventas_especiales
         WHERE DATE(fecha_y_hora) = ?
           AND sucursal_id = ?
         ORDER BY fecha_y_hora DESC
         LIMIT ? OFFSET ?";
$stmtE = $conn->prepare($sqlE);
$stmtE->bind_param("siii", $hoy, $sucursal_id, $limitE, $offsetE);
$stmtE->execute();
$resE = $stmtE->get_result();
$ventas_especiales = [];
while ($rowE = $resE->fetch_assoc()) {
    $ventas_especiales[] = $rowE;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Listado de Ventas de Hoy</title>
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
  <style>
    body {
      background-color: #F2F2F2;
      font-family: Arial, sans-serif;
    }
    .table td, .table th {
      vertical-align: middle;
    }
    .detalle-venta-row {
      display: none; /* se mostrará vía JS */
      background-color: #fafafa;
    }
    .detalle-venta-row td {
      border-top: none !important;
    }
    .pagination {
      margin-top: 10px;
    }
  </style>
</head>
<body class="p-3">

<h2>Ventas del Día (Sucursal: <?php echo htmlspecialchars($sucursalName, ENT_QUOTES, 'UTF-8'); ?>)</h2>
<a href="index_simplificado.php" class="btn btn-secondary">Regresar a VENTAS</a>
<p>Fecha: <?php echo date('d/m/Y'); ?></p>

<hr>

<!-- ======================================== -->
<!--   SECCIÓN: VENTAS NORMALES (paginadas)  -->
<!-- ======================================== -->
<h3>Ventas Normales</h3>
<?php if ($totalN === 0): ?>
  <p>No hay ventas normales registradas hoy.</p>
<?php else: ?>
  <table class="table table-bordered">
    <thead>
      <tr>
        <th>ID</th>
        <th>Fecha/Hora</th>
        <th>Total</th>
        <th>Acción</th>
      </tr>
    </thead>
    <tbody>
    <?php foreach ($ventas_normales as $venta): ?>
      <tr>
        <td><?php echo $venta['id']; ?></td>
        <td><?php echo date('d/m/Y H:i', strtotime($venta['fecha_y_hora'])); ?></td>
        <td>Q <?php echo number_format($venta['total'], 2); ?></td>
        <td>
          <button class="btn btn-sm btn-info ver-detalle" data-id="<?php echo $venta['id']; ?>">
            Ver Detalle
          </button>
        </td>
      </tr>
      <!-- Fila oculta para mostrar detalle -->
      <tr class="detalle-venta-row" id="detalle-venta-<?php echo $venta['id']; ?>">
        <td colspan="4">
          <div class="detalle-content"></div>
        </td>
      </tr>
    <?php endforeach; ?>
    </tbody>
  </table>

  <!-- Paginación VENTAS NORMALES -->
  <nav>
    <ul class="pagination justify-content-center">
      <!-- Botón Previous -->
      <?php if ($pageN > 1): 
        $prevN = $pageN - 1; ?>
        <li class="page-item">
          <a class="page-link" href="?pageN=<?php echo $prevN; ?>&pageE=<?php echo $pageE; ?>">&laquo;</a>
        </li>
      <?php else: ?>
        <li class="page-item disabled">
          <span class="page-link">&laquo;</span>
        </li>
      <?php endif; ?>

      <?php
      $max_displayN = 5;
      $startN = max(1, $pageN - floor($max_displayN / 2));
      $endN = min($totalPagesN, $startN + $max_displayN - 1);
      if ($endN - $startN + 1 < $max_displayN) {
          $startN = max(1, $endN - $max_displayN + 1);
      }
      for ($i = $startN; $i <= $endN; $i++):
      ?>
        <?php if ($i == $pageN): ?>
          <li class="page-item active">
            <span class="page-link"><?php echo $i; ?></span>
          </li>
        <?php else: ?>
          <li class="page-item">
            <a class="page-link" href="?pageN=<?php echo $i; ?>&pageE=<?php echo $pageE; ?>">
              <?php echo $i; ?>
            </a>
          </li>
        <?php endif; ?>
      <?php endfor; ?>

      <!-- Botón Next -->
      <?php if ($pageN < $totalPagesN):
        $nextN = $pageN + 1; ?>
        <li class="page-item">
          <a class="page-link" href="?pageN=<?php echo $nextN; ?>&pageE=<?php echo $pageE; ?>">&raquo;</a>
        </li>
      <?php else: ?>
        <li class="page-item disabled">
          <span class="page-link">&raquo;</span>
        </li>
      <?php endif; ?>
    </ul>
  </nav>
<?php endif; ?>

<hr>

<!-- ======================================== -->
<!--   SECCIÓN: VENTAS ESPECIALES (paginadas) -->
<!-- ======================================== -->
<h3>Ventas Especiales</h3>
<?php if ($totalE === 0): ?>
  <p>No hay ventas especiales registradas hoy.</p>
<?php else: ?>
  <table class="table table-bordered">
    <thead>
      <tr>
        <th>ID</th>
        <th>Fecha/Hora</th>
        <th>Descripción</th>
        <th>Precio</th>
      </tr>
    </thead>
    <tbody>
    <?php foreach ($ventas_especiales as $vesp): ?>
      <tr>
        <td><?php echo $vesp['id']; ?></td>
        <td><?php echo date('d/m/Y H:i', strtotime($vesp['fecha_y_hora'])); ?></td>
        <td><?php echo htmlspecialchars($vesp['descripcion'], ENT_QUOTES, 'UTF-8'); ?></td>
        <td>Q <?php echo number_format($vesp['precio'], 2); ?></td>
      </tr>
    <?php endforeach; ?>
    </tbody>
  </table>

  <!-- Paginación VENTAS ESPECIALES -->
  <nav>
    <ul class="pagination justify-content-center">
      <!-- Botón Previous -->
      <?php if ($pageE > 1):
        $prevE = $pageE - 1; ?>
        <li class="page-item">
          <a class="page-link" href="?pageN=<?php echo $pageN; ?>&pageE=<?php echo $prevE; ?>">&laquo;</a>
        </li>
      <?php else: ?>
        <li class="page-item disabled">
          <span class="page-link">&laquo;</span>
        </li>
      <?php endif; ?>

      <?php
      $max_displayE = 5;
      $startE = max(1, $pageE - floor($max_displayE / 2));
      $endE = min($totalPagesE, $startE + $max_displayE - 1);
      if ($endE - $startE + 1 < $max_displayE) {
          $startE = max(1, $endE - $max_displayE + 1);
      }
      for ($j = $startE; $j <= $endE; $j++):
      ?>
        <?php if ($j == $pageE): ?>
          <li class="page-item active">
            <span class="page-link"><?php echo $j; ?></span>
          </li>
        <?php else: ?>
          <li class="page-item">
            <a class="page-link" href="?pageN=<?php echo $pageN; ?>&pageE=<?php echo $j; ?>">
              <?php echo $j; ?>
            </a>
          </li>
        <?php endif; ?>
      <?php endfor; ?>

      <!-- Botón Next -->
      <?php if ($pageE < $totalPagesE):
        $nextE = $pageE + 1; ?>
        <li class="page-item">
          <a class="page-link" href="?pageN=<?php echo $pageN; ?>&pageE=<?php echo $nextE; ?>">&raquo;</a>
        </li>
      <?php else: ?>
        <li class="page-item disabled">
          <span class="page-link">&raquo;</span>
        </li>
      <?php endif; ?>
    </ul>
  </nav>
<?php endif; ?>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
// =============================
// Mostrar/Ocultar detalle
// =============================
$(document).ready(function() {
  // Al hacer clic en "Ver Detalle" (ventas normales)
  $('.ver-detalle').on('click', function() {
    const ventaId = $(this).data('id');
    const rowDetalle = $('#detalle-venta-' + ventaId);
    const content = rowDetalle.find('.detalle-content');

    // Si ya está visible, lo ocultamos
    if (rowDetalle.is(':visible')) {
      rowDetalle.hide();
      return;
    }

    // Si no está visible, hacemos AJAX para cargar el detalle
    $.ajax({
      url: 'get_detalle_venta.php',
      method: 'GET',
      data: { venta_id: ventaId },
      success: function(htmlDetalle) {
        content.html(htmlDetalle);
        rowDetalle.show();
      },
      error: function() {
        content.html('<p style="color:red;">Error al cargar el detalle.</p>');
        rowDetalle.show();
      }
    });
  });
});
</script>

</body>
</html>
